<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\GoogleAuthenticator;
use App\Models\AdminNotification;
use App\Models\ApplyService;
use App\Models\Deposit;
use App\Models\DeviceToken;
use App\Models\Form;
use App\Models\Service;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller {
    public function home() {
        $pageTitle = 'Dashboard';

        $user      = auth()->user();
        $latestTrx = Transaction::where('user_id', $user->id)->latest()->take('10')->get();

        $widgets['total_deposit']               = Deposit::where('user_id', $user->id)->where('status', 1)->sum('amount');
        $widgets['total_trx']                   = Transaction::where('user_id', $user->id)->count();
        $widgets['count_apply_service']         = ApplyService::where('user_id', $user->id)->where('status', '!=', 0)->count();
        $widgets['count_apply_service_pending'] = ApplyService::where('user_id', $user->id)->where('status', 2)->count();
        $widgets['downline']                    = User::where('ref_by', $user->id)->count();

        return view('Template::user.dashboard', compact('pageTitle', 'user', 'latestTrx', 'widgets'));
    }

    public function depositHistory(Request $request) {
        $pageTitle = 'Deposit History';
        $deposits  = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.deposit_history', compact('pageTitle', 'deposits'));
    }

    public function show2faForm() {
        $ga        = new GoogleAuthenticator();
        $user      = auth()->user();
        $secret    = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . gs('site_name'), $secret);
        $pageTitle = '2FA Security';
        return view('Template::user.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request) {
        $user = auth()->user();
        $request->validate([
            'key'  => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($user, $request->code, $request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts  = Status::ENABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator activated successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }

    public function disable2fa(Request $request) {
        $request->validate([
            'code' => 'required',
        ]);

        $user     = auth()->user();
        $response = verifyG2fa($user, $request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts  = Status::DISABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator deactivated successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function transactions() {
        $pageTitle = 'Transactions';
        $remarks   = Transaction::distinct('remark')->orderBy('remark')->get('remark');

        $transactions = Transaction::where('user_id', auth()->id())->searchable(['trx'])->filter(['trx_type', 'remark'])->orderBy('id', 'desc')->paginate(getPaginate());

        return view('Template::user.transactions', compact('pageTitle', 'transactions', 'remarks'));
    }

    public function userData() {
        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $pageTitle  = 'User Data';
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = @implode(',', $info['code']);
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));

        return view('Template::user.user_data', compact('pageTitle', 'user', 'countries', 'mobileCode'));
    }

    public function userDataSubmit(Request $request) {

        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $countryData  = (array) json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'username'     => 'required|unique:users|min:6',
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('users')->where('dial_code', $request->mobile_code)],
        ]);

        if (preg_match("/[^a-z0-9_]/", trim($request->username))) {
            $notify[] = ['info', 'Username can contain only small letters, numbers and underscore.'];
            $notify[] = ['error', 'No special character, space or capital letters in username.'];
            return back()->withNotify($notify)->withInput($request->all());
        }

        $user->country_code = $request->country_code;
        $user->mobile       = $request->mobile;
        $user->username     = $request->username;

        $user->address      = $request->address;
        $user->city         = $request->city;
        $user->state        = $request->state;
        $user->zip          = $request->zip;
        $user->country_name = @$request->country;
        $user->dial_code    = $request->mobile_code;

        $user->profile_complete = Status::YES;
        $user->save();

        return to_route('user.home');
    }

    public function addDeviceToken(Request $request) {

        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return ['success' => false, 'errors' => $validator->errors()->all()];
        }

        $deviceToken = DeviceToken::where('token', $request->token)->first();

        if ($deviceToken) {
            return ['success' => true, 'message' => 'Already exists'];
        }

        $deviceToken          = new DeviceToken();
        $deviceToken->user_id = auth()->user()->id;
        $deviceToken->token   = $request->token;
        $deviceToken->is_app  = Status::NO;
        $deviceToken->save();

        return ['success' => true, 'message' => 'Token saved successfully'];
    }

    public function downloadAttachment($fileHash) {

        $filePath  = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $title     = slug(gs('site_name')) . '- attachments.' . $extension;
        try
        {
            $mimetype = mime_content_type($filePath);
        } catch (\Exception $e) {
            $notify[] = ['error', 'File does not exists'];
            return back()->withNotify($notify);
        }
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function downlines() {
        $pageTitle = 'Downline';
        $user      = auth()->user();
        $downlines = User::where('ref_by', $user->id)->latest()->paginate(getPaginate());
        return view('Template::user.downlines', compact('pageTitle', 'downlines'));
    }

    public function services() {
        $pageTitle = 'Services';
        $services  = Service::where('status', 1)->whereHas('category', function ($query) {
            $query->where('status', 1);
        })->paginate(getPaginate());

        return view('Template::user.service.services', compact('pageTitle', 'services'));
    }

    public function serviceApply(Request $request) {

        $request->validate([
            'id'     => 'required|exists:services',
            'amount' => 'required|numeric|gt:0',
        ]);

        $findService = Service::where('id', $request->id)
            ->where('status', 1)
            ->whereHas('category', function ($query) {
                $query->where('status', 1);
            })->firstOrFail();

        $chargeFree     = 0;
        $requiredBalnce = 0;
        $user           = auth()->user();
        $charge         = 0;
        $amount         = $request->amount;

        if ($chargeFree == $findService->fixed_charge + $findService->parcent_charge) {
            $requiredBalnce = $amount;
        } else {
            $charge         = $findService->fixed_charge + ($amount * $findService->percent_charge / 100);
            $requiredBalnce = $amount + $charge;
        }

        if ($user->balance < $requiredBalnce) {
            $notify[] = ['error', 'Sorry insufficient balance'];
            return redirect()->route('user.deposit.index')->withNotify($notify);
        }

        $apply               = new ApplyService();
        $apply->user_id      = $user->id;
        $apply->service_id   = $findService->id;
        $apply->amount       = $amount;
        $apply->total_charge = $charge;
        $apply->after_charge = $requiredBalnce;
        $apply->save();

        return redirect()->route('user.service.confirm', $apply->id);
    }

    public function serviceConfirm($id) {

        $apply = ApplyService::where('status', 0)
            ->where('id', $id)
            ->where('user_id', auth()->user()->id)
            ->firstOrFail();

        $service = Service::where('id', $apply->service_id)
            ->where('status', 1)
            ->whereHas('category', function ($query) {
                $query->where('status', 1);
            })->firstOrFail();

        $pageTitle = 'Service Confirmation';
        $user      = auth()->user();

        return view('Template::user.service.confirmService', compact('pageTitle', 'apply', 'service', 'user'));
    }

    public function serviceSubmit(Request $request) {
        $user  = auth()->user();
        $apply = ApplyService::where('status', 0)
            ->where('user_id', $user->id)
            ->where('id', $request->apply_id)
            ->firstOrFail();

        $findService = Service::where('id', $apply->service_id)
            ->where('status', 1)
            ->whereHas('category', function ($query) {
                $query->where('status', 1);
            })->firstOrFail();

        $customValidation       = [];
        $customValidation['id'] = 'required|exists:apply_services,id';
        $selectField            = [];

        if ($findService->select_field && $findService->category->field_name) {
            $selectArray     = json_decode($findService->select_field, true);
            $selectFieldName = array_keys($selectArray);
            $selectFieldName = implode(' ', $selectFieldName);
            $selectValues    = $selectArray[$selectFieldName];
            $selectValues    = implode(',', $selectValues);

            $selectField[$selectFieldName] = $request->$selectFieldName;

            $customValidation[$selectFieldName] = 'required|in:' . $selectValues;
        }

        $formData       = $findService->form->form_data;
        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $userData = $formProcessor->processFormData($request, $formData);

        if ($user->balance < $apply->after_charge) {
            $notify[] = ['error', 'Sorry insufficient balance'];
            return redirect()->route('user.deposit')->withNotify($notify);
        }

        $user->balance -= $apply->after_charge;
        $user->save();

        $apply->status       = 2;
        $apply->post_balance = $user->balance;
        $apply->select_field = $selectField ? json_encode($selectField) : null;
        $apply->user_data    = json_encode($userData);
        $apply->save();

        $general = gs();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = getAmount($apply->amount);
        $transaction->post_balance = getAmount($user->balance);
        $transaction->charge       = getAmount($apply->total_charge);
        $transaction->trx_type     = '-';
        $transaction->details      = 'Requested for ' . $apply->service->category->name . ' service';
        $transaction->remark       = 'service_requested';
        $transaction->trx          = getTrx();

        $transaction->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = $transaction->details . ' from ' . $user->username;
        $adminNotification->click_url = route('admin.applied.pending.service');
        $adminNotification->save();

        notify($user, 'SERVICE_APPLY_COMPLETE', [
            'trx'          => $transaction->trx,
            'amount'       => $apply->amount,
            'charge'       => $apply->total_charge,
            'currency'     => $general->cur_text,
            'post_balance' => $user->balance,
            'service_name' => $apply->service->category->name,
        ]);

        $notify[] = ['success', 'Service applied successfully'];
        return redirect()->route('user.history.service')->withNotify($notify);
    }

    public function serviceHistory() {
        $pageTitle = 'Service History';
        $histories = ApplyService::where('user_id', auth()->user()->id)->where('status', '!=', 0)->latest()->paginate(getPaginate());
        return view('Template::user.service.history', compact('pageTitle', 'histories'));
    }

    public function serviceHistoryPending() {
        $pageTitle = 'Pending Service History';
        $histories = ApplyService::where('user_id', auth()->user()->id)->where('status', 2)->latest()->paginate(getPaginate());
        return view('Template::user.service.history', compact('pageTitle', 'histories'));
    }
}
